'use client';

import { useState, useEffect } from 'react';
import Image from 'next/image';

interface ProductImageProps {
  src: string;
  alt: string;
  fill?: boolean;
  width?: number;
  height?: number;
  sizes?: string;
  className?: string;
  onError?: (e: React.SyntheticEvent<HTMLImageElement, Event>) => void;
  onLoad?: () => void;
  priority?: boolean; // For above-the-fold images
}

/**
 * Optimized Product Image Component - FAST loading
 * Shows images immediately without blocking conversions
 */
export default function ProductImage({
  src,
  alt,
  fill = false,
  width,
  height,
  sizes,
  className = '',
  onError,
  onLoad,
  priority = false,
}: ProductImageProps) {
  const [imageUrl, setImageUrl] = useState<string>(src || '');
  const [imageLoaded, setImageLoaded] = useState(false);

  useEffect(() => {
    if (!src || typeof src !== 'string') {
      setImageUrl('');
      return;
    }

    const trimmed = src.trim();
    setImageUrl(trimmed);
    
    // Try to convert ImgBB gallery URLs in background (non-blocking)
    if (trimmed.includes('ibb.co') && !trimmed.includes('i.ibb.co')) {
      // Show original URL immediately, convert in background
      fetch(`/api/imgbb-convert?url=${encodeURIComponent(trimmed)}`)
        .then(res => res.json())
        .then(data => {
          if (data.success && data.directUrl) {
            setImageUrl(data.directUrl);
          }
        })
        .catch(() => {
          // Keep original URL on error
        });
    }
  }, [src]);

  const handleLoad = () => {
    setImageLoaded(true);
    if (onLoad) onLoad();
  };

  // Use regular img tag for external URLs (http/https/data URLs) - FASTEST
  if (imageUrl && (imageUrl.startsWith('http://') || imageUrl.startsWith('https://') || imageUrl.startsWith('data:'))) {
    if (fill) {
      return (
        <div className="relative w-full h-full max-w-full">
          {!imageLoaded && (
            <div className="absolute inset-0 bg-gradient-to-br from-primary-100 to-sky-100 animate-pulse" />
          )}
          <img
            src={imageUrl}
            alt={alt}
            className={`${className} ${imageLoaded ? 'opacity-100' : 'opacity-0'} transition-opacity duration-300 max-w-full h-auto`}
            onError={onError}
            onLoad={handleLoad}
            loading={priority ? "eager" : "lazy"}
            decoding="async"
            crossOrigin={imageUrl.startsWith('data:') ? undefined : "anonymous"}
            style={{ width: '100%', height: '100%', objectFit: 'cover', maxWidth: '100%' }}
            fetchPriority={priority ? "high" : "auto"}
          />
        </div>
      );
    }
    
    return (
      <div className="relative max-w-full">
        {!imageLoaded && (
          <div className="absolute inset-0 bg-gradient-to-br from-primary-100 to-sky-100 animate-pulse" />
        )}
        <img
          src={imageUrl}
          alt={alt}
          className={`${className} ${imageLoaded ? 'opacity-100' : 'opacity-0'} transition-opacity duration-300 max-w-full h-auto`}
          width={width}
          height={height}
          onError={onError}
          onLoad={handleLoad}
          loading={priority ? "eager" : "lazy"}
          decoding="async"
          crossOrigin={imageUrl.startsWith('data:') ? undefined : "anonymous"}
          fetchPriority={priority ? "high" : "auto"}
          style={{ maxWidth: '100%', height: 'auto' }}
        />
      </div>
    );
  }

  // Use Next.js Image for local paths
  if (fill) {
    return (
      <Image
        src={imageUrl || '/images/placeholder.jpg'}
        alt={alt}
        fill
        className={className}
        sizes={sizes}
        onError={onError}
        onLoad={handleLoad}
        priority={priority}
        unoptimized
      />
    );
  }

  return (
    <Image
      src={imageUrl || '/images/placeholder.jpg'}
      alt={alt}
      width={width || 400}
      height={height || 300}
      className={className}
      sizes={sizes}
      onError={onError}
      onLoad={handleLoad}
      priority={priority}
      unoptimized
    />
  );
}

