'use client';

import { motion } from 'framer-motion';
import { useInView } from 'framer-motion';
import { useRef, useState, useEffect } from 'react';
import { Mail, Phone, MapPin, Send, CheckCircle, RefreshCw, Shield } from 'lucide-react';
import { saveFormSubmission } from '@/lib/admin';

export default function ContactHomeSection() {
  const ref = useRef(null);
  const isInView = useInView(ref, { once: true, margin: '-100px' });
  const [isMounted, setIsMounted] = useState(false);
  
  useEffect(() => {
    setIsMounted(true);
  }, []);
  
  const shouldAnimate = isMounted || isInView;
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    message: '',
  });
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);
  
  // CAPTCHA
  const generateCaptcha = () => {
    const num1 = Math.floor(Math.random() * 10) + 1;
    const num2 = Math.floor(Math.random() * 10) + 1;
    return { num1, num2, answer: '' };
  };
  const [captcha, setCaptcha] = useState<{ num1: number; num2: number; answer: string } | null>(null);
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
    setCaptcha(generateCaptcha());
  }, []);

  const validate = () => {
    const newErrors: Record<string, string> = {};
    
    // Validate name
    if (!formData.name.trim()) {
      newErrors.name = 'Name is required';
    } else if (formData.name.trim().length < 2) {
      newErrors.name = 'Name must be at least 2 characters';
    } else if (!/^[a-zA-Z\s]+$/.test(formData.name.trim())) {
      newErrors.name = 'Name can only contain letters and spaces';
    }
    
    // Validate email
    if (!formData.email.trim()) {
      newErrors.email = 'Email is required';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email.trim())) {
      newErrors.email = 'Please enter a valid email address';
    }
    
    // Validate phone
    if (!formData.phone.trim()) {
      newErrors.phone = 'Phone number is required';
    } else {
      const cleanedPhone = formData.phone.replace(/[\s\-+]/g, '');
      if (!/^[6-9]\d{9}$/.test(cleanedPhone)) {
        newErrors.phone = 'Please enter a valid 10-digit Indian phone number';
      }
    }
    
    // Validate message
    if (!formData.message.trim()) {
      newErrors.message = 'Message is required';
    } else if (formData.message.trim().length < 10) {
      newErrors.message = 'Message must be at least 10 characters';
    }
    
    // Validate CAPTCHA
    if (!captcha) {
      newErrors.captcha = 'Please wait for captcha to load';
    } else if (!captcha.answer.trim()) {
      newErrors.captcha = 'Please solve the captcha';
    } else if (parseInt(captcha.answer) !== captcha.num1 + captcha.num2) {
      newErrors.captcha = 'Incorrect answer. Please try again.';
    }
    
    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!validate()) {
      return;
    }
    
    setIsSubmitting(true);
    setErrors({}); // Clear previous errors
    
    try {
      // Save form submission to admin dashboard (API)
      const result = await saveFormSubmission({
        source: 'contact-home',
        data: {
          name: formData.name,
          email: formData.email,
          phone: formData.phone,
          message: formData.message,
        },
      });
      
      if (result.success) {
        setIsSubmitted(true);
        setFormData({ name: '', email: '', phone: '', message: '' });
        setErrors({});
        setCaptcha(generateCaptcha());
        setTimeout(() => setIsSubmitted(false), 5000);
      } else {
        // Show error to user
        setErrors({ submit: result.error || 'Failed to submit form. Please try again.' });
        console.error('Form submission failed:', result.error);
      }
    } catch (error: any) {
      console.error('Error submitting form:', error);
      setErrors({ submit: error.message || 'Failed to submit form. Please try again.' });
    } finally {
      // Always reset submitting state, even if there's an error
      setIsSubmitting(false);
    }
  };
  
  const handleCaptchaChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (captcha) {
      setCaptcha({ ...captcha, answer: e.target.value });
      if (errors.captcha) {
        setErrors({ ...errors, captcha: '' });
      }
    }
  };
  
  const resetCaptcha = () => {
    const newCaptcha = generateCaptcha();
    setCaptcha({ ...newCaptcha, answer: captcha?.answer || '' });
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData({
      ...formData,
      [name]: value,
    });
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors({ ...errors, [name]: '' });
    }
  };

  return (
    <div>
    <section ref={ref} className="section-padding bg-gradient-to-br from-white via-sky-50/40 to-primary-50/30">
      <div className="container mx-auto px-4 sm:px-6 md:px-8">
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={shouldAnimate ? { opacity: 1, y: 0 } : { opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
          className="text-center mb-8 sm:mb-10 md:mb-12"
        >
          <h2 className="text-2xl sm:text-3xl md:text-4xl lg:text-5xl font-bold bg-gradient-to-r from-primary-700 via-primary-600 to-accent-600 bg-clip-text text-transparent mb-3 sm:mb-4 px-2 sm:px-0">Contact Us</h2>
          <p className="text-base sm:text-lg text-slate-600 max-w-2xl mx-auto font-medium px-2 sm:px-0">
            Get in touch with us for all your wooden packaging needs
          </p>
        </motion.div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 sm:gap-10 md:gap-12">
          {/* Contact Form */}
          <motion.div
            initial={{ opacity: 0, x: -50 }}
            animate={shouldAnimate ? { opacity: 1, x: 0 } : { opacity: 1, x: 0 }}
            transition={{ duration: 0.8, delay: 0.2 }}
            className="bg-white rounded-lg shadow-lg p-5 sm:p-6 md:p-8 order-2 lg:order-1"
          >
            <h3 className="text-2xl sm:text-3xl font-bold text-wooden-900 mb-4 sm:mb-5 md:mb-6">Send us a Message</h3>
            
            {isSubmitted && (
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                className="mb-6 p-4 bg-gradient-to-r from-green-50 to-emerald-50 border-2 border-green-400 text-green-800 rounded-xl flex items-center space-x-3 shadow-lg"
              >
                <CheckCircle size={24} className="text-green-600 flex-shrink-0" />
                <div>
                  <p className="font-bold text-green-800">Success!</p>
                  <p className="text-sm text-green-700">Message sent successfully! We'll get back to you soon.</p>
                </div>
              </motion.div>
            )}

            <form onSubmit={handleSubmit} className="space-y-6">
              <div>
                <label htmlFor="name" className="block text-slate-800 font-bold mb-2">
                  Name *
                </label>
                <input
                  type="text"
                  id="name"
                  name="name"
                  value={formData.name}
                  onChange={handleChange}
                  className={`w-full px-4 py-3 border-2 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-400 outline-none transition-all ${
                    errors.name ? 'border-red-500 bg-red-50' : 'border-slate-200 focus:border-primary-400'
                  }`}
                  placeholder="Your Name"
                />
                {errors.name && (
                  <p className="mt-1 text-sm text-red-600 font-medium">{errors.name}</p>
                )}
              </div>

              <div>
                <label htmlFor="email" className="block text-slate-800 font-bold mb-2">
                  Email *
                </label>
                <input
                  type="email"
                  id="email"
                  name="email"
                  value={formData.email}
                  onChange={handleChange}
                  className={`w-full px-4 py-3 border-2 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-400 outline-none transition-all ${
                    errors.email ? 'border-red-500 bg-red-50' : 'border-slate-200 focus:border-primary-400'
                  }`}
                  placeholder="your.email@example.com"
                />
                {errors.email && (
                  <p className="mt-1 text-sm text-red-600 font-medium">{errors.email}</p>
                )}
              </div>

              <div>
                <label htmlFor="phone" className="block text-slate-800 font-bold mb-2">
                  Phone *
                </label>
                <input
                  type="tel"
                  id="phone"
                  name="phone"
                  value={formData.phone}
                  onChange={handleChange}
                  className={`w-full px-4 py-3 border-2 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-400 outline-none transition-all ${
                    errors.phone ? 'border-red-500 bg-red-50' : 'border-slate-200 focus:border-primary-400'
                  }`}
                  placeholder="+91 96509 75030 or 9650975030"
                />
                {errors.phone && (
                  <p className="mt-1 text-sm text-red-600 font-medium">{errors.phone}</p>
                )}
              </div>

              <div>
                <label htmlFor="message" className="block text-slate-800 font-bold mb-2">
                  Message *
                </label>
                <textarea
                  id="message"
                  name="message"
                  rows={5}
                  value={formData.message}
                  onChange={handleChange}
                  className={`w-full px-4 py-3 border-2 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-400 outline-none transition-all resize-none ${
                    errors.message ? 'border-red-500 bg-red-50' : 'border-slate-200 focus:border-primary-400'
                  }`}
                  placeholder="Tell us about your packaging requirements..."
                />
                {errors.message && (
                  <p className="mt-1 text-sm text-red-600 font-medium">{errors.message}</p>
                )}
              </div>

              {/* CAPTCHA */}
              {mounted && captcha && (
                <div>
                  <label htmlFor="captcha" className="flex items-center gap-2 text-slate-800 font-bold mb-2">
                    <Shield size={18} className="text-primary-600" />
                    Human Verification <span className="text-red-500">*</span>
                  </label>
                  <div className="flex items-center gap-3">
                    <div className="flex items-center space-x-2 px-4 py-2.5 bg-gradient-to-r from-primary-50 to-sky-50 border-2 border-primary-200 rounded-xl">
                      <span className="text-xl font-bold text-slate-900">
                        {captcha.num1} + {captcha.num2} = ?
                      </span>
                    </div>
                    <input
                      type="number"
                      id="captcha"
                      name="captcha"
                      value={captcha.answer}
                      onChange={handleCaptchaChange}
                      className={`flex-1 px-4 py-2.5 border-2 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-400 outline-none transition-all ${
                        errors.captcha ? 'border-red-500 bg-red-50' : 'border-slate-200'
                      }`}
                      placeholder="Answer"
                    />
                    <button
                      type="button"
                      onClick={resetCaptcha}
                      className="p-2.5 bg-slate-100 hover:bg-slate-200 rounded-xl transition-colors border border-slate-300"
                      aria-label="Refresh captcha"
                    >
                      <RefreshCw size={18} className="text-slate-700" />
                    </button>
                  </div>
                  {errors.captcha && (
                    <p className="mt-1 text-sm text-red-600 font-medium">{errors.captcha}</p>
                  )}
                </div>
              )}

              {errors.submit && (
                <div className="p-4 bg-red-50 border-2 border-red-400 text-red-800 rounded-xl">
                  <p className="font-bold text-red-800">Error</p>
                  <p className="text-sm text-red-700">{errors.submit}</p>
                </div>
              )}

              <button
                type="submit"
                disabled={isSubmitting}
                className="w-full btn-primary flex items-center justify-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isSubmitting ? (
                  <>
                    <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
                    <span>Sending...</span>
                  </>
                ) : (
                  <>
                    <Send size={20} />
                    <span>Send Message</span>
                  </>
                )}
              </button>
            </form>
          </motion.div>

          {/* Contact Info */}
            <motion.div
              initial={{ opacity: 0, x: 50 }}
              animate={shouldAnimate ? { opacity: 1, x: 0 } : { opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.4 }}
              className="space-y-6 sm:space-y-8 bg-gradient-to-br from-primary-50/50 to-sky-50/50 rounded-2xl p-5 sm:p-6 md:p-8 border-2 border-primary-100 shadow-xl order-1 lg:order-2"
            >
              <div>
                <h3 className="text-2xl sm:text-3xl font-bold bg-gradient-to-r from-primary-700 to-primary-600 bg-clip-text text-transparent mb-4 sm:mb-5 md:mb-6">Get in Touch</h3>
                <p className="text-base sm:text-lg text-slate-700 mb-6 sm:mb-7 md:mb-8 font-medium">
                  We're here to help you with all your wooden packaging needs. Reach out to us 
                  through any of the following channels.
                </p>
              </div>

              <div className="space-y-4 sm:space-y-5 md:space-y-6">
                <div className="flex items-start space-x-3 sm:space-x-4 bg-white p-3 sm:p-4 rounded-xl shadow-md border border-primary-100">
                  <div className="w-10 h-10 sm:w-12 sm:h-12 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl flex items-center justify-center flex-shrink-0 shadow-lg">
                    <Phone className="text-white" size={18} />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="text-lg sm:text-xl font-bold text-slate-800 mb-1">Phone</h4>
                    <a href="tel:+919650975030" className="text-primary-600 hover:text-primary-700 transition-colors block font-semibold text-sm sm:text-base">
                      +91 96509 75030
                    </a>
                    <a href="tel:+919990214636" className="text-primary-600 hover:text-primary-700 transition-colors block font-semibold text-sm sm:text-base">
                      +91 99902 14636
                    </a>
                  </div>
                </div>

                <div className="flex items-start space-x-3 sm:space-x-4 bg-white p-3 sm:p-4 rounded-xl shadow-md border border-primary-100">
                  <div className="w-10 h-10 sm:w-12 sm:h-12 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl flex items-center justify-center flex-shrink-0 shadow-lg">
                    <Mail className="text-white" size={18} />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="text-lg sm:text-xl font-bold text-slate-800 mb-1">Email</h4>
                    <a href="mailto:info@sspackaging.com" className="text-primary-600 hover:text-primary-700 transition-colors font-semibold text-sm sm:text-base break-all">
                      info@sspackaging.com
                    </a>
                  </div>
                </div>

                <div className="flex items-start space-x-3 sm:space-x-4 bg-white p-3 sm:p-4 rounded-xl shadow-md border border-primary-100">
                  <div className="w-10 h-10 sm:w-12 sm:h-12 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl flex items-center justify-center flex-shrink-0 shadow-lg">
                    <MapPin className="text-white" size={18} />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="text-lg sm:text-xl font-bold text-slate-800 mb-1">Address</h4>
                    <p className="text-slate-600 font-medium text-sm sm:text-base">
                      2-B-334, Sector 9 OPP B-123 Sector 10,<br />
                      Noida, Uttar Pradesh 201301<br />
                      India
                    </p>
                  </div>
                </div>
              </div>

              {/* Map */}
              <div className="mt-6 sm:mt-8 rounded-xl overflow-hidden shadow-lg border-2 border-primary-100">
                <a 
                  href="https://maps.app.goo.gl/bPMGCBXaLNWekn8Z9" 
                  target="_blank" 
                  rel="noopener noreferrer"
                  className="block"
                >
                  <iframe
                    src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3504.304492890623!2d77.3258648!3d28.5899652!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x390ce5c6cafe88f5%3A0x7492a0b342a2cdf8!2sSharma%20Arts%20Sec-9!5e0!3m2!1sen!2sin!4v1736500000000!5m2!1sen!2sin"
                    width="100%"
                    height="250"
                    style={{ border: 0 }}
                    allowFullScreen
                    loading="lazy"
                    referrerPolicy="no-referrer-when-downgrade"
                    className="w-full h-56 sm:h-64 pointer-events-none"
                  ></iframe>
                </a>
                <div className="mt-2 text-center bg-primary-50 py-2">
                  <a 
                    href="https://maps.app.goo.gl/bPMGCBXaLNWekn8Z9" 
                    target="_blank" 
                    rel="noopener noreferrer"
                    className="text-primary-600 hover:text-primary-700 text-sm font-semibold underline inline-flex items-center gap-1"
                  >
                    <MapPin size={16} />
                    View on Google Maps
                  </a>
                </div>
              </div>
            </motion.div>
        </div>
      </div>
    </section>
    </div>
  );
}
